"""notebooksalamode.Datasets"""

from six import string_types, PY2
import hashlib
import sys

class Datasets(object):
    """Container providing multiple lookup methods for each registered Dataset.

    ... by Dataset name
    ... by Dataset token
    ... by Dataset insert order to provide backwards compatibility with early
        versions of our notebooks.
    """
    def __init__(self):
        self.by_index = []
        self.by_token = {}
        self.by_name = {}

    def __getitem__(self, key):
        if isinstance(key, string_types):
            # for python2, upshift to unicode if needed
            if PY2 and isinstance(key, str):
                key = key.decode("utf-8")

            if key in self.by_token:
                return self.by_token[key].get_data()

            hkey = hashlib.sha256(key.encode('utf-8')).hexdigest()
            if hkey in self.by_name:
                return self.by_name[hkey].get_data()
            raise KeyError
        else:
            if key >= 0 and key <= len(self.by_index):
                return self.by_index[key].get_data()
            raise IndexError

    def add_dataset(self, dset):
        """Add the provided dataset"""
        self.by_index.append(dset)
        self.by_token[dset.token] = dset
        self.by_name[hashlib.sha256(dset.name.encode('utf-8')).hexdigest()] = dset

    def _repr_html_(self):
        output = (
            u"<table class='dataframe'>\n"
            u"<thead>\n"
            u"<tr style='text-align: right'>"
            u"<th>index</th>"
            u"<th>qref</th>"
            u"<th>name</th>"
            u"<th>rows</th>"
            u"<th>columns</th>"
            u"</tr></thead>\n"
            u"<tbody>\n")

        for i, dset in enumerate(self.by_index):
            rows, cols = dset.get_data_shape()
            style = u"text-color: #ccc" if dset.error else u""
            if dset.error:
                output += u"<tr style='{}'><td>{}</td><td>{}</td><td>{}</td><td>{}</td><td>{}</td></tr>\n".format(
                    style, i, dset.token, dset.name, rows, cols)
            else:
                output += u"<tr><td>{}</td><td>{}</td><td>{}</td><td>{}</td><td>{}</td></tr>\n".format(
                    i, dset.token, dset.name, rows, cols)

        output += u"</tbody>\n</table>"
        return output
