import time
from ipykernel.comm import Comm

class CommStringIO(object):
    """ CommStringIO """
    def __init__(self, name=None):
        self.comm = Comm(target_name=name)
        self.buffer = []
        self.maxbuf = 10000
        self.CHUNK_SIZE = 512 * 1024 * 1024 # 2GB is max size for a message, so use 512MB to be safe

    def close(self):
        """ Closes the CommStringIO. """
        self.flush()
        self.comm.close()

    def isatty(self):
        """Returns false. Just for compatibility to IO."""
        return False

    def __iter__(self):
        """
        Raises exception. Just for compatibility to IO.
        See https://github.com/pandas-dev/pandas/blob/3c0cf22/pandas/core/dtypes/inference.py#L186
        """
        raise Exception('__iter__ called unexpectedly in CommStringIO')

    def write(self, newstr):
        """Appends the given string to the underlying buffer string. Buffer is flushed
           every maxbuf entries."""
        self.buffer.append(newstr)
        if len(self.buffer) >= self.maxbuf:
            self.flush()

    def flush(self):
        """Flushes any buffer entries"""
        if not self.buffer:
            return

        # 1. Create the massive string
        full_data = ''.join(self.buffer)
        total_len = len(full_data)
        
        # 2. Slice it up
        for i in range(0, total_len, self.CHUNK_SIZE):
            chunk = full_data[i:i + self.CHUNK_SIZE]
            
            # 3. Send as a standard message
            # ipykernel automatically wraps this in {"comm_id": "...", "data": chunk}
            # This ensures 'NewWireMessageFromBytes' in Go does not fail.
            self.comm.send(data=chunk)
            
            # Critical: Give Go a tiny moment to process the handler
            # This prevents flooding the Go channel/buffer
            time.sleep(0.05)

        self.buffer = []
