/*
 * Decompiled with CFR 0.152.
 */
package com.becon.opencelium.backend.scriptengine.external.polyglotservice;

import com.becon.opencelium.backend.constant.props.PolyglotProps;
import com.becon.opencelium.backend.scriptengine.external.polyglotservice.PolyglotServiceManager;
import jakarta.annotation.PreDestroy;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.stereotype.Component;

@Component
public class PolyglotServiceManager
implements DisposableBean {
    private static final Logger logger = LoggerFactory.getLogger(PolyglotServiceManager.class);
    private final PolyglotProps props;
    private Process polyglotProcess;
    private volatile boolean isRunning = false;

    public PolyglotServiceManager(PolyglotProps props) {
        this.props = props;
        this.registerShutdownHook();
    }

    private void registerShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            if (this.isRunning) {
                logger.info("Shutdown hook triggered - stopping Polyglot service");
                this.cleanup();
            }
        }, "polyglot-shutdown-hook"));
    }

    public void start() throws PolyglotStartupException {
        if (!this.props.isEnabled() || !this.props.isAutoStart()) {
            logger.info("Polyglot service auto-start is disabled");
            return;
        }
        if (this.isRunning) {
            logger.warn("Polyglot service is already running");
            return;
        }
        logger.info("Starting Polyglot service from: {}", (Object)this.props.getJarPath());
        try {
            this.polyglotProcess = this.startProcess();
            this.waitForServiceStart();
            this.isRunning = true;
            logger.info("Polyglot service started successfully on {}://{}:{}", new Object[]{this.props.getProtocol(), this.props.getHost(), this.props.getPort()});
        }
        catch (Exception e) {
            this.cleanup();
            throw new PolyglotStartupException("Failed to start Polyglot service", (Throwable)e);
        }
    }

    private Process startProcess() throws IOException {
        List command = this.buildCommand();
        logger.debug("Starting process with command: {}", (Object)String.join((CharSequence)" ", command));
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        processBuilder.redirectErrorStream(true);
        Process process = processBuilder.start();
        if (this.props.isExternalLogEnabled()) {
            this.startOutputLogger(process);
        }
        return process;
    }

    private List<String> buildCommand() {
        ArrayList<String> command = new ArrayList<String>();
        command.add("java");
        if (this.props.getJvmArgs() != null && !this.props.getJvmArgs().trim().isEmpty()) {
            String[] jvmArgs;
            for (String arg : jvmArgs = this.props.getJvmArgs().trim().split("\\s+")) {
                command.add(arg);
            }
        }
        command.add("-jar");
        command.add(this.props.getJarPath());
        if (this.props.getArgs() != null && !this.props.getArgs().trim().isEmpty()) {
            String[] args;
            for (String arg : args = this.props.getArgs().trim().split("\\s+")) {
                command.add(arg);
            }
        }
        return command;
    }

    private void startOutputLogger(Process process) {
        Thread loggerThread = new Thread(() -> {
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    logger.info("[Polyglot] {}", (Object)line);
                }
            }
            catch (IOException e) {
                logger.error("Error reading Polyglot process output", (Throwable)e);
            }
        });
        loggerThread.setDaemon(true);
        loggerThread.setName("polyglot-output-logger");
        loggerThread.start();
    }

    private void waitForServiceStart() throws PolyglotStartupException {
        logger.info("Waiting for Polyglot service to start (timeout: {}s)", (Object)this.props.getWaitTimeoutSec());
        long startTime = System.currentTimeMillis();
        long timeoutMillis = (long)this.props.getWaitTimeoutSec() * 1000L;
        while (System.currentTimeMillis() - startTime < timeoutMillis) {
            if (!this.polyglotProcess.isAlive()) {
                throw new PolyglotStartupException("Polyglot process terminated unexpectedly");
            }
            if (this.isPortListening()) {
                logger.info("Polyglot service port is available at {}:{}", (Object)this.props.getHost(), (Object)this.props.getPort());
                return;
            }
            try {
                Thread.sleep(1000L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new PolyglotStartupException("Startup wait interrupted", (Throwable)e);
            }
        }
        throw new PolyglotStartupException("Polyglot service failed to start within " + this.props.getWaitTimeoutSec() + " seconds");
    }

    private boolean isPortListening() {
        try {
            Socket socket = new Socket();
            socket.connect(new InetSocketAddress(this.props.getHost(), (int)this.props.getPort()), 2000);
            socket.close();
            return true;
        }
        catch (Exception e) {
            logger.trace("Port {}:{} not available yet: {}", new Object[]{this.props.getHost(), this.props.getPort(), e.getMessage()});
            return false;
        }
    }

    public boolean isRunning() {
        return this.isRunning && this.polyglotProcess != null && this.polyglotProcess.isAlive();
    }

    @PreDestroy
    public void destroy() {
        this.stop();
    }

    public void stop() {
        if (this.polyglotProcess == null || !this.isRunning) {
            return;
        }
        logger.info("Stopping Polyglot service...");
        this.cleanup();
        logger.info("Polyglot service stopped");
    }

    private void cleanup() {
        this.isRunning = false;
        if (this.polyglotProcess == null) {
            return;
        }
        try {
            this.polyglotProcess.destroy();
            boolean terminated = this.polyglotProcess.waitFor(10L, TimeUnit.SECONDS);
            if (!terminated) {
                logger.warn("Polyglot service did not terminate gracefully, forcing shutdown");
                this.polyglotProcess.destroyForcibly();
                this.polyglotProcess.waitFor(5L, TimeUnit.SECONDS);
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            logger.error("Interrupted while stopping Polyglot service", (Throwable)e);
            this.polyglotProcess.destroyForcibly();
        }
        finally {
            this.polyglotProcess = null;
        }
    }
}

